"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireWildcard(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _uriPattern = _interopRequireWildcard(require("./uri-pattern"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _stubItem = _interopRequireDefault(require("../items/stub-item"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * PaneItem registers an opener with the current Atom workspace as long as this component is mounted. The opener will
 * trigger on URIs that match a specified pattern and render a subtree returned by a render prop.
 *
 * The render prop can receive three arguments:
 *
 * * itemHolder: A RefHolder. If used as the target for a ref, the referenced component will be used as the "item" of
 *   the pane item - its `getTitle()`, `getIconName()`, and other methods will be used by the pane.
 * * params: An object containing the named parameters captured by the URI pattern.
 * * uri: The exact, matched URI used to launch this item.
 *
 * render() {
 *   return (
 *     <PaneItem workspace={this.props.workspace} uriPattern='atom-github://host/{id}'>
 *       {({itemHolder, params}) => (
 *         <ItemComponent ref={itemHolder.setter} id={params.id} />
 *       )}
 *     </PaneItem>
 *   );
 * }
 */
class PaneItem extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'opener');
    const uriPattern = new _uriPattern.default(this.props.uriPattern);
    const currentlyOpen = this.props.workspace.getPaneItems().reduce((arr, item) => {
      const element = item.getElement ? item.getElement() : null;
      const match = item.getURI ? uriPattern.matches(item.getURI()) : _uriPattern.nonURIMatch;
      const stub = item.setRealItem ? item : null;

      if (element && match.ok()) {
        const openItem = new OpenItem(match, element, stub);
        arr.push(openItem);
      }

      return arr;
    }, []);
    this.subs = new _eventKit.CompositeDisposable();
    this.state = {
      uriPattern,
      currentlyOpen
    };
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.uriPattern.getOriginal() === nextProps.uriPattern) {
      return null;
    }

    return {
      uriPattern: new _uriPattern.default(nextProps.uriPattern)
    };
  }

  componentDidMount() {
    // Listen for and adopt StubItems that are added after this component has
    // already been mounted.
    this.subs.add(this.props.workspace.onDidAddPaneItem(({
      item
    }) => {
      if (!item._getStub) {
        return;
      }

      const stub = item._getStub();

      if (stub.getRealItem() !== null) {
        return;
      }

      const match = this.state.uriPattern.matches(item.getURI());

      if (!match.ok()) {
        return;
      }

      const openItem = new OpenItem(match, stub.getElement(), stub);
      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });

      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }

      this.registerCloseListener(item, openItem);
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }));
    }));

    for (const openItem of this.state.currentlyOpen) {
      this.registerCloseListener(openItem.stubItem, openItem);
      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });

      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }
    }

    this.subs.add(this.props.workspace.addOpener(this.opener));
  }

  render() {
    return this.state.currentlyOpen.map(item => {
      return _react.default.createElement(_react.Fragment, {
        key: item.getKey()
      }, item.renderPortal(this.props.children));
    });
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  opener(uri) {
    const m = this.state.uriPattern.matches(uri);

    if (!m.ok()) {
      return undefined;
    }

    const openItem = new OpenItem(m);

    if (this.props.className) {
      openItem.addClassName(this.props.className);
    }

    return new Promise(resolve => {
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }), () => {
        const paneItem = openItem.create({
          copy: () => this.copyOpenItem(openItem)
        });
        this.registerCloseListener(paneItem, openItem);
        resolve(paneItem);
      });
    });
  }

  copyOpenItem(openItem) {
    const m = this.state.uriPattern.matches(openItem.getURI());

    if (!m.ok()) {
      return null;
    }

    const stub = _stubItem.default.create('generic', openItem.getStubProps(), openItem.getURI());

    const copiedItem = new OpenItem(m, stub.getElement(), stub);
    this.setState(prevState => ({
      currentlyOpen: [...prevState.currentlyOpen, copiedItem]
    }), () => {
      this.registerCloseListener(stub, copiedItem);
      copiedItem.hydrateStub({
        copy: () => this.copyOpenItem(copiedItem)
      });
    });
    return stub;
  }

  registerCloseListener(paneItem, openItem) {
    const sub = this.props.workspace.onDidDestroyPaneItem(({
      item
    }) => {
      if (item === paneItem) {
        sub.dispose();
        this.subs.remove(sub);
        this.setState(prevState => ({
          currentlyOpen: prevState.currentlyOpen.filter(each => each !== openItem)
        }));
      }
    });
    this.subs.add(sub);
  }

}
/**
 * A subtree rendered through a portal onto a detached DOM node for use as the root as a PaneItem.
 */


exports.default = PaneItem;

_defineProperty(PaneItem, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  children: _propTypes.default.func.isRequired,
  uriPattern: _propTypes.default.string.isRequired,
  className: _propTypes.default.string
});

class OpenItem {
  constructor(match, element = null, stub = null) {
    this.id = this.constructor.nextID;
    this.constructor.nextID++;
    this.domNode = element || document.createElement('div');
    this.domNode.tabIndex = '-1';
    this.domNode.onfocus = this.onFocus.bind(this);
    this.stubItem = stub;
    this.stubProps = stub ? stub.props : {};
    this.match = match;
    this.itemHolder = new _refHolder.default();
  }

  getURI() {
    return this.match.getURI();
  }

  create(extra = {}) {
    const h = this.itemHolder.isEmpty() ? null : this.itemHolder;
    return (0, _helpers.createItem)(this.domNode, h, this.match.getURI(), extra);
  }

  hydrateStub(extra = {}) {
    if (this.stubItem) {
      this.stubItem.setRealItem(this.create(extra));
      this.stubItem = null;
    }
  }

  addClassName(className) {
    this.domNode.classList.add(className);
  }

  getKey() {
    return this.id;
  }

  getStubProps() {
    const itemProps = this.itemHolder.map(item => ({
      title: item.getTitle ? item.getTitle() : null,
      iconName: item.getIconName ? item.getIconName() : null
    }));
    return _objectSpread2({}, this.stubProps, {}, itemProps);
  }

  onFocus() {
    return this.itemHolder.map(item => item.focus && item.focus());
  }

  renderPortal(renderProp) {
    return _reactDom.default.createPortal(renderProp({
      deserialized: this.stubProps,
      itemHolder: this.itemHolder,
      params: this.match.getParams(),
      uri: this.match.getURI()
    }), this.domNode);
  }

}

_defineProperty(OpenItem, "nextID", 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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